<?php
/**
 * Contains PhpMyAdmin\Plugins\Schema\Svg\RelationStatsSvg class
 */

declare(strict_types=1);

namespace PhpMyAdmin\Plugins\Schema\Svg;

use PhpMyAdmin\ConfigStorage\Relation;
use PhpMyAdmin\Identifiers\DatabaseName;
use PhpMyAdmin\Plugins\Schema\ExportRelationSchema;
use PhpMyAdmin\Version;

use function __;
use function in_array;
use function max;
use function min;
use function sprintf;

/**
 * RelationStatsSvg Relation Schema Class
 *
 * Purpose of this class is to generate the SVG XML Document because
 * SVG defines the graphics in XML format which is used for representing
 * the database diagrams as vector image. This class actually helps
 *  in preparing SVG XML format.
 *
 * SVG XML is generated by using XMLWriter php extension and this class
 * inherits ExportRelationSchema class has common functionality added
 * to this class
 */
class SvgRelationSchema extends ExportRelationSchema
{
    /** @var TableStatsSvg[] */
    private array $tables = [];

    /** @var RelationStatsSvg[] Relations */
    private array $relations = [];

    private int|float $xMax = 0;

    private int|float $yMax = 0;

    private int|float $xMin = 100000;

    private int|float $yMin = 100000;

    private int|float $tablewidth = 0;

    private Svg $svg;

    /**
     * Upon instantiation This starts writing the SVG XML document
     * user will be prompted for download as .svg extension
     *
     * @see Svg
     */
    public function __construct(Relation $relation, DatabaseName $db)
    {
        parent::__construct($relation, $db);

        $this->svg = new Svg();

        $this->setShowColor(isset($_REQUEST['svg_show_color']));
        $this->setShowKeys(isset($_REQUEST['svg_show_keys']));
        $this->setTableDimension(isset($_REQUEST['svg_show_table_dimension']));
        $this->setAllTablesSameWidth(isset($_REQUEST['svg_all_tables_same_width']));

        $this->svg->setTitle(
            sprintf(
                __('Schema of the %s database - Page %s'),
                $this->db->getName(),
                $this->pageNumber,
            ),
        );
        $this->svg->setAuthor('phpMyAdmin ' . Version::VERSION);
        $this->svg->setFont('Arial');
        $this->svg->setFontSize(16);

        $alltables = $this->getTablesFromRequest();

        foreach ($alltables as $table) {
            if (! isset($this->tables[$table])) {
                $this->tables[$table] = new TableStatsSvg(
                    $this->svg,
                    $this->db->getName(),
                    $table,
                    $this->svg->getFont(),
                    $this->svg->getFontSize(),
                    $this->pageNumber,
                    $this->showKeys,
                    $this->tableDimension,
                    $this->offline,
                );
                $this->tablewidth = max($this->tablewidth, $this->tables[$table]->width);
            }

            if ($this->sameWide) {
                $this->tables[$table]->width = &$this->tablewidth;
            }

            $this->setMinMax($this->tables[$table]);
        }

        $border = 15;
        $this->svg->startSvgDoc(
            $this->xMax + $border,
            $this->yMax + $border,
            $this->xMin - $border,
            $this->yMin - $border,
        );

        $seenARelation = false;
        foreach ($alltables as $oneTable) {
            $existRel = $this->relation->getForeignersInternal($this->db->getName(), $oneTable);
            if ($existRel === []) {
                continue;
            }

            $seenARelation = true;
            foreach ($existRel as $masterField => $rel) {
                // put the foreign table on the schema only if selected by the user
                if (! in_array($rel['foreign_table'], $alltables, true)) {
                    continue;
                }

                $this->addRelation(
                    $oneTable,
                    $this->svg->getFont(),
                    $this->svg->getFontSize(),
                    $masterField,
                    $rel['foreign_table'],
                    $rel['foreign_field'],
                    $this->tableDimension,
                );
            }

            foreach ($this->relation->getForeignKeysData($this->db->getName(), $oneTable) as $oneKey) {
                if (! in_array($oneKey->refTableName, $alltables, true)) {
                    continue;
                }

                foreach ($oneKey->indexList as $index => $oneField) {
                    $this->addRelation(
                        $oneTable,
                        $this->svg->getFont(),
                        $this->svg->getFontSize(),
                        $oneField,
                        $oneKey->refTableName,
                        $oneKey->refIndexList[$index],
                        $this->tableDimension,
                    );
                }
            }
        }

        if ($seenARelation) {
            $this->drawRelations();
        }

        $this->drawTables();
        $this->svg->endSvgDoc();
    }

    /** @return array{fileName: non-empty-string, fileData: string} */
    public function getExportInfo(): array
    {
        return ['fileName' => $this->getFileName('.svg'), 'fileData' => $this->svg->getOutputData()];
    }

    /**
     * Sets X and Y minimum and maximum for a table cell
     *
     * @param TableStatsSvg $table The table
     */
    private function setMinMax(TableStatsSvg $table): void
    {
        $this->xMax = max($this->xMax, $table->x + $table->width);
        $this->yMax = max($this->yMax, $table->y + $table->height);
        $this->xMin = min($this->xMin, $table->x);
        $this->yMin = min($this->yMin, $table->y);
    }

    /**
     * Defines relation objects
     *
     * @see setMinMax,TableStatsSvg::__construct(),
     *       PhpMyAdmin\Plugins\Schema\Svg\RelationStatsSvg::__construct()
     *
     * @param string $masterTable    The master table name
     * @param string $font           The font face
     * @param int    $fontSize       Font size
     * @param string $masterField    The relation field in the master table
     * @param string $foreignTable   The foreign table name
     * @param string $foreignField   The relation field in the foreign table
     * @param bool   $tableDimension Whether to display table position or not
     */
    private function addRelation(
        string $masterTable,
        string $font,
        int $fontSize,
        string $masterField,
        string $foreignTable,
        string $foreignField,
        bool $tableDimension,
    ): void {
        if (! isset($this->tables[$masterTable])) {
            $this->tables[$masterTable] = new TableStatsSvg(
                $this->svg,
                $this->db->getName(),
                $masterTable,
                $font,
                $fontSize,
                $this->pageNumber,
                false,
                $tableDimension,
            );
            $this->tablewidth = max($this->tablewidth, $this->tables[$masterTable]->width);
            $this->setMinMax($this->tables[$masterTable]);
        }

        if (! isset($this->tables[$foreignTable])) {
            $this->tables[$foreignTable] = new TableStatsSvg(
                $this->svg,
                $this->db->getName(),
                $foreignTable,
                $font,
                $fontSize,
                $this->pageNumber,
                false,
                $tableDimension,
            );
            $this->tablewidth = max($this->tablewidth, $this->tables[$foreignTable]->width);
            $this->setMinMax($this->tables[$foreignTable]);
        }

        $this->relations[] = new RelationStatsSvg(
            $this->svg,
            $this->tables[$masterTable],
            $masterField,
            $this->tables[$foreignTable],
            $foreignField,
        );
    }

    /**
     * Draws relation arrows and lines
     * connects master table's master field to
     * foreign table's foreign field
     *
     * @see Relation_Stats_Svg::relationDraw()
     */
    private function drawRelations(): void
    {
        foreach ($this->relations as $relation) {
            $relation->relationDraw($this->showColor);
        }
    }

    /**
     * Draws tables
     */
    private function drawTables(): void
    {
        foreach ($this->tables as $table) {
            $table->tableDraw($this->showColor);
        }
    }
}
